##########################################################################
## An Introduction to Empirical Legal Studies
## 	Chapter 10 Replication File
##
## Revised on 3/21/2014 KEC	  
##
## This work is licensed under a Creative Commons Attribution 4.0
## International License. See http://creativecommons.org/licenses/by/4.0
## for more information.
##
## Authors
##	Lee Epstein, Washington University in St. Louis
##  and Andrew D. Martin, University of Michigan
##########################################################################

## Chapter 10

## load packages and set up multiplot 
library(foreign)
library(ggplot2)
library(reshape)
library(grid) 
source("ggplotTemplate.R")

multiplot <- function(..., plotlist=NULL, file, cols=1, layout=NULL) {
  require(grid)

  # Make a list from the ... arguments and plotlist
  plots <- c(list(...), plotlist)

  numPlots = length(plots)

  # If layout is NULL, then use 'cols' to determine layout
  if (is.null(layout)) {
    # Make the panel
    # ncol: Number of columns of plots
    # nrow: Number of rows needed, calculated from # of cols
    layout <- matrix(seq(1, cols * ceiling(numPlots/cols)),
                    ncol = cols, nrow = ceiling(numPlots/cols))
  }

 if (numPlots==1) {
    print(plots[[1]])

  } else {
    # Set up the page
    grid.newpage()
    pushViewport(viewport(layout = grid.layout(nrow(layout), ncol(layout))))

    # Make each plot, in the correct location
    for (i in 1:numPlots) {
      # Get the i,j matrix positions of the regions that contain this subplot
      matchidx <- as.data.frame(which(layout == i, arr.ind = TRUE))

      print(plots[[i]], vp = viewport(layout.pos.row = matchidx$row,
                                      layout.pos.col = matchidx$col))
    }
  }
}

## Figure 10.1 Dot plots of the length (in words) of present-day constitutions
## in 41 European countries, by European region. If the goal is to provide readers 
## with exact figures, Table 10.3 is the better display. If the goal is to convey 
## patterns or trends, as it usually is, we prefer this figure to Table 10.3.
## Section 10.1, p. 243

## read in data
ccp <- read.dta("ccpCharacteristics.dta")
ccp$country_new[99] <- "Macedonia"
ccpEurope <- ccp[is.na(ccp$europe)==FALSE,];summary(ccpEurope$europe)
#ccpEurope <- ccpEurope[order(ccpEurope$europe,decreasing=TRUE),]
#ccpEurope <- ccpEurope[order(ccpEurope$length_n,decreasing=TRUE),]
ccpEurope <- ccpEurope[order(ccpEurope$europe,ccpEurope$length_n,decreasing=TRUE),]
ccpEurope$country_new <- factor(ccpEurope$country_new, levels=unique(as.character(ccpEurope$country_new)) )

lineData <- data.frame(europe=c(rep("Eastern Europe",10),rep("Northern Europe",9),
                                rep("Southern Europe",13),rep("Western Europe",9)),lineLocations=c(1:10,1:9,1:13,1:9))

dpEurope <- ggplot(ccpEurope, aes(x=length_n, y=country_new)) + geom_point(size=3) + 
  geom_hline(aes(yintercept=lineLocations),linetype=3,lineData) +
xlab("Length of Constitution") + ylab("") + ggtitle("") + 
  facet_grid(europe~.,scale="free",space="free")

## show plot and save it
print(dpEurope)
ggsave("dpEurope.pdf", dpEurope, height=8, width=6)

## Figure 10.2 Length (in words) of present-day constitutions in countries in 
## Southern Europe. Although it is possible from the table to observe the 
## patteners that jump out of the graph--for example, the large difference 
## between the shortest and longest constitutions--it requires far more 
## (unnecessary) congnitive work
## Section 10.1, p. 245

ccp$country_new[99] <- "Macedonia"
ccpEurope <- ccp[is.na(ccp$europe)==FALSE,];summary(ccpEurope$europe)
ccpSEurope <- ccpEurope[ccpEurope$europe=="Southern Europe",];
summary(ccpSEurope$europe)
sortSEurope <- ccpSEurope[order(ccpSEurope$length_n,decreasing=TRUE),]
sortSEurope$country_new <- factor(sortSEurope$country_new, 
                                  levels=unique(as.character(sortSEurope$country_new)) )

dpSouthernOrdered <- ggplot(sortSEurope, aes(x=length_n, y=country_new)) + 
  geom_point(size=3) +
  geom_hline(yintercept=1, linetype=3) +
  geom_hline(yintercept=2, linetype=3) +
  geom_hline(yintercept=3, linetype=3) +
  geom_hline(yintercept=4, linetype=3) +
  geom_hline(yintercept=5, linetype=3) +
  geom_hline(yintercept=6, linetype=3) +
  geom_hline(yintercept=7, linetype=3) +
  geom_hline(yintercept=8, linetype=3) +
  geom_hline(yintercept=9, linetype=3) +
  geom_hline(yintercept=10, linetype=3) +
  geom_hline(yintercept=11, linetype=3) +
  geom_hline(yintercept=12, linetype=3) +
  geom_hline(yintercept=13, linetype=3) +
 scale_x_continuous(limits=c(0,40000),breaks=c(0,10000,20000,30000,40000)) +
  xlab("Length of Constitution") + ylab("");dpSouthernOrdered

## show plot and save it
print(dpSouthernOrdered)
ggsave("dpSouthernOrdered.pdf", dpSouthernOrdered, height=5, width=5)

## Figure 10.4 Both panels depict the mean life expectancy at birth (in years)
## in five regions, by gender. The bottom panel reflects our attempt to reduce 
## clutter. Specifically, we eliminated (1) the depth cue, (2) the internal data 
## labels, (3) tick marks on the horizontal axis, and (4) the legend. We also 
## reduced the number of tick marks on the vertical axis and supplanted the 
## cross hatching with a solid color. The result is a graph far easier to decode
## Section 10.2, p. 249

## read in data and transform it
leData <- read.dta("lifeExpectancyWorld.dta")
femaleMeans <- tapply(leData$female_life, list(continent=leData$continent), mean)
maleMeans <- tapply(leData$male_life, list(continent=leData$continent), mean)
leCombined <- t(rbind(maleMeans,femaleMeans))
dimnames(leCombined) <- list(c("Africa","Americas","Asia","Europe","Oceania"),
                             c("Males","Females"))
leMelt <- melt(leCombined)

## make plot
cloudPlot <- ggplot(leMelt, aes(y=value, x=X1)) + geom_bar(stat="identity") +
  coord_flip(ylim=c(50, 82))+ facet_grid(~X2, scales="free") + 
  labs(y="Average Life Expectancy",x="") + 
  theme(panel.margin = unit(2, "lines"),axis.ticks.y = element_blank())

## show plot and save it
print(cloudPlot)
ggsave("cloudPlot.pdf", cloudPlot, height=4, width=6)

## Figure 10.5 Both panels depict the number of criminal and civil cases filed
## per authorized judgeship in the U.S. Court of Appeals, 1992-2010. The bottom
## panel reflects our attempt to improve visualization of the data. Specifically 
## we eliminated the legend (replacing it with internal data labels) and the dark 
## grid, both of which obscure the data. We also reduced the number of tick marks 
## and associated labels. Finally, we supplanted the non-circular subelements 
## with circular connectors, though if the author's purpose is merely to show 
## trends in new filings, the circles may be unnecessary
## Section 10.2, p. 250

## generate the data
caseloadData <- read.table(header=T,text='
year	Civil	Criminal
1992	20716	11215
1993	21639	11862
1994	21218	10674
1995	21630	10162
1996	21279	10889
1997	21198	10521
1998	22055	10535
1999	23971	10251
2000	23501	10707
2001	24540	11281
2002	21925	11569
2003	21200	11968
2004	21440	12506
2005	21666	16060
2006	21494	15246
2007	19389	13167
2008	19001	13667
2009	19211	13710
2010	19655	12689
')
caseloadMelt <- melt(caseloadData,id="year")

## create the lineplot
linePlot <- ggplot(caseloadMelt, aes(x=year, y=value,group=variable,
                                     shape=variable,size=variable)) + 
  geom_line(aes(size = variable)) + scale_size_manual(values=c(1,.5))
linePlot <- linePlot + geom_point(size=2)+scale_shape_manual(values=c(19,1)) +
  xlim(1992,2012)
linePlot <- linePlot + xlab("Year") + ylab("Cases Filed Per Authorized Judgeship")+ 
  theme(legend.position = "none") 
linePlot <- linePlot + annotate("text",x=2011, y=19655,label="Civil",
                                family="Times",size=4) 
linePlot <- linePlot + annotate("text",x=2011.5, y=12689, label="Criminal",
                                family="Times",size=4)

## show plot and save it
print(linePlot)
ggsave("linePlot.pdf", linePlot, height=5, width=9)

## Figure 10.6 Dot plots of the length (in words) of present-day constitutions
## in Eastern Europe. From the top panel it is difficult to observe several 
## potentially interesting patterns in the data, such as the gap between Bulgaria 
## and Hungary. We excluded zero in the bottom panel to facilitate a more effective 
## judgment about the data
## Section 10.2, p. 254

## read in data and generate variables 
ccp <- read.dta("ccpCharacteristics.dta")
ccp$country_new[99] <- "Macedonia"
ccpEurope <- ccp[is.na(ccp$europe)==FALSE,];summary(ccpEurope$europe)
ccpEEurope <- ccpEurope[ccpEurope$europe=="Eastern Europe",]
sortEEurope <- ccpEEurope[order(ccpEEurope$length_n,decreasing=TRUE),]
sortEEurope$country_new <- factor(sortEEurope$country_new, 
                                  levels=unique(as.character(sortEEurope$country_new)) )

## make first plot
dpEasternZero <- ggplot(sortEEurope, aes(x=length_n, y=country_new)) + 
  geom_point(size=3) +
  geom_hline(yintercept=1, linetype=3) +
  geom_hline(yintercept=2, linetype=3) +
  geom_hline(yintercept=3, linetype=3) +
  geom_hline(yintercept=4, linetype=3) +
  geom_hline(yintercept=5, linetype=3) +
  geom_hline(yintercept=6, linetype=3) +
  geom_hline(yintercept=7, linetype=3) +
  geom_hline(yintercept=8, linetype=3) +
  geom_hline(yintercept=9, linetype=3) +
  geom_hline(yintercept=10, linetype=3) +
   scale_x_continuous(limits=c(0,20000),breaks=c(0,5000,10000,15000,20000)) +
  xlab("") + ylab("") + ggtitle("Eastern Europe") + theme(plot.margin = 
                                                            unit(c(.5,1,-.75,0), "lines"))

## show first plot and save it
print(dpEasternZero)
ggsave("dpEasternZero.pdf", dpEasternZero, height=2, width=6)

## make second plot
dpEasternNoZero <- ggplot(sortEEurope, aes(x=length_n, y=country_new)) + 
  geom_point(size=3) +
  geom_hline(yintercept=1, linetype=3) +
  geom_hline(yintercept=2, linetype=3) +
  geom_hline(yintercept=3, linetype=3) +
  geom_hline(yintercept=4, linetype=3) +
  geom_hline(yintercept=5, linetype=3) +
  geom_hline(yintercept=6, linetype=3) +
  geom_hline(yintercept=7, linetype=3) +
  geom_hline(yintercept=8, linetype=3) +
  geom_hline(yintercept=9, linetype=3) +
  geom_hline(yintercept=10, linetype=3) +
   scale_x_continuous(limits=c(10000,20000),breaks=c(10000,12000,14000,
                                                     16000,18000,20000)) +
  xlab("") + ylab("") + ggtitle("") + theme(plot.margin = unit(c(-1,1,0,0), 
                                                               "lines"))

## show second plot and save it
print(dpEasternNoZero)
ggsave("dpEasternNoZero.pdf", dpEasternNoZero, height=2, width=6)

## show plot and save it
EasternEuropeZero1 <- multiplot(dpEasternZero, dpEasternNoZero,cols=1)
print(EasternEuropeZero1)
pdf("EasternEuropeZero1.pdf", EasternEuropeZero1, height=6, width=7)

## Figure 10.7 All three panels show the Trust in Government Index (developed 
## from the American National Election Studies) on the vertical axis and year 
## (from 1958-2008) on the horizonal axis. From the top panel it is difficult 
# to observe the decline in trust over time. We excluded zero in the middle 
## pattern to facilitate a more effective judgment about the data, and did not 
## connect 1958 to 1964 to accentuate data sparseness. A problem in both the 
## top and middle panels, however, is that neither clearly delineates missing 
## years. To fix it, we moved to a bar chart. Now readers can spot missing data 
## it is also easier to see that the Index is available only for even-numbered 
## years. The sequence of charts shores up Cleveland's advice: "Don't hesitate to 
## make two or more graphs of the same data."
## Section 10.2, p. 256

## read in data
schroederDataBar <- read.table(header=T, text='
year value
1964	52
1966	61
1968	45
1970	39
1972	38
1974	29
1976	30
1978	29
1980	27
1982	31
1984	38
1986	47
1988	34
1990	29
1992	29
1994	26
1996	32
1998	34
2000	36
2002  43
2004  37
')

## make the top figure
linePlotSchroeder1 <- ggplot(schroederDataBar, aes(x=year, y=value)) + 
  geom_line(size=.5)
linePlotSchroeder1 <- linePlotSchroeder1 + geom_point(size=4,shape=18)+ 
  scale_x_continuous(limits=c(1957,2009),breaks=c(1958,1960,1962,1964,1966,
                                                  1968,1970,1972,1974,1976,
                                                  1978,1980,1982,1984,1986,
                                                  1988,1990,1992,1994,1996,
                                                  1998,2000,2002,2004,2006,2008), 
                     labels=c("1958","1960","1962","1964","1966","1968","1970",
                              "1972","1974","1976","1978","1980","1982","1984",
                              "1986","1988","1990","1992","1994","1996","1998",
                              "2000","2002","2004","2006","2008")) + 
  scale_y_continuous(limits=c(0,70),breaks=c(0,10,20,30,40,50,60,70),
                     labels=c("0","","20","","40","","60","")) + 
  theme(axis.text.x= element_text(angle=45, hjust = 1),
        panel.grid.major.x = element_blank(),panel.grid.major.y =
          element_blank(),panel.grid.minor.x = element_blank(),panel.grid.minor.y = 
          element_blank())
linePlotSchroeder1 <- linePlotSchroeder1 + xlab("") + ylab("")+ 
  theme(legend.position = "none") 

## make middle plot
linePlotSchroeder2 <- ggplot(schroederDataBar, aes(x=year, y=value)) + 
  geom_line(size=.5)
linePlotSchroeder2 <- linePlotSchroeder2 + geom_point(size=4,shape=18)+ 
  xlim(1957,2009) + geom_point(size=4,shape=18,aes(x=1958,y=49)) + 
  geom_point(size=4,aes(x=2008,y=26),shape=18)
linePlotSchroeder2 <- linePlotSchroeder2 + xlab("") + ylab("")+ 
  theme(legend.position = "none") + theme(panel.grid.major.x = 
                                            element_blank(),panel.grid.major.y =
                                            element_line(size=.2, 
                                                         color="gray",linetype="dotted"),
                                          panel.grid.minor.x = element_blank(),
                                          panel.grid.minor.y = element_blank())

## make the bottom plot
barPlotSchroeder <- ggplot(schroederDataBar, aes(x=year, y=value)) + 
  geom_bar(width=1.5,stat="identity")
barPlotSchroeder <- barPlotSchroeder + xlab("") + ylab("")+ 
  theme(legend.position = "none") 
barPlotSchroeder <- barPlotSchroeder + scale_x_continuous(breaks=c(1958,1964,1966,
                                                                   1968,1970,1972,
                                                                   1974,1976,1978,
                                                                   1980,1982,1984,
                                                                   1986,1988,1990,
                                                                   1992,1994,1996,
                                                                   1998,2000,2002,
                                                                   2004,2008), 
                                                          labels=c("1958","1964",
                                                                   "1966","1968",
                                                                   "1970","1972",
                                                                   "1974","1976",
                                                                   "1978","1980",
                                                                   "1982","1984",
                                                                   "1986","1988",
                                                                   "1990","1992",
                                                                   "1994","1996",
                                                                   "1998","2000",
                                                                   "2002","2004",
                                                                   "2008")) + 
  theme(axis.text.x= element_text(angle=45, size=8),panel.grid.major.x = 
          element_blank(),panel.grid.major.y =element_line(size=1, color="gray",
                                                           linetype="dotted"),
        panel.grid.minor.x = element_blank(),panel.grid.minor.y = element_blank())

## put all three together
trustInGovt<-multiplot(linePlotSchroeder1, linePlotSchroeder2, barPlotSchroeder, 
                       cols=1)

## show multiplot and save it
print(trustInGovt)
pdf("trustInGovt.pdf", trustInGovt)
